/* $Id: libstress.c,v 1.8 1998/09/16 02:05:11 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Attempt to stress the ability of the E1432 host interface library
   to deal with multiple calls to init, uninit, assign, create, and so
   on.  Look for file descriptor leeks, inability to handle empty
   groups, and similar problems.

   Also check for a known problem with e1432_preset. */

#include <stdlib.h>		/* For exit, strtol */
#include <stdio.h>		/* For printf */
#include <string.h>		/* For strrchr */
#include <time.h>		/* For clock */
#include <unistd.h>		/* For getopt */
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024

/* The number of loops is arbitrary, but keep it greater than 60
   to flush out anything which uses up file descriptors */
#define	LOOPS	100

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: libstress.c,v 1.8 1998/09/16 02:05:11 ericb Exp $";
static const char *progname;

static int
stress(int nmod, SHORTSIZ16 *laddr, int verbose)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    SHORTSIZ16 chan_list[4] = { E1432_INPUT_CHAN(1),
				E1432_INPUT_CHAN(2),
				E1432_INPUT_CHAN(3),
				E1432_INPUT_CHAN(4) };
    E1432ID hw;
    int     i, group, chan;

    CHECK(e1432_display_state(0));

    /* We used to have a bug in e1432_preset which would cause the
       following sequence to hang a Sonata SCA, if there is a tach
       board present. */
    if (verbose > 0)
	(void) printf("Testing e1432_preset bug\n");
    CHECK(e1432_init_io_driver());
    CHECK(e1432_get_hwconfig(1, laddr, hwconfig));
    if (hwconfig[0].tach_chans > 0)
	chan = E1432_TACH_CHAN(1);
    else
	chan = E1432_INPUT_CHAN(1);
    CHECK(e1432_assign_channel_numbers(1, laddr, &hw));
    CHECK(e1432_preset(hw, chan));
    CHECK(e1432_init_measure(hw, chan));
    CHECK(e1432_assign_channel_numbers(1, laddr, &hw));

    if (verbose > 0)
	(void) printf("Testing e1432_init_io_driver\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    if (verbose > 0)
	(void) printf("Testing e1432_assign_channel_numbers\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_assign_channel_numbers(nmod, laddr, &hw));
    if (verbose > 0)
	(void) printf("Testing e1432_get_hwconfig\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));
    if (verbose > 0)
	(void) printf("Testing e1432_create_channel_group\n");

    for (i = 0; i < LOOPS; i++)
    {
	group = e1432_create_channel_group(hw, 4, chan_list);
	if (group >= 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_create_channel_group: returned %d\n",
			   progname, group);
	    return -1;
	}
	if (group < 0 && (i & 0x1) == 0)
	    CHECK(e1432_delete_channel_group(hw, group));

	group = e1432_create_channel_group(hw, 0, NULL);
	if (group >= 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_create_channel_group: returned %d\n",
			   progname, group);
	    return -1;
	}
	if (group < 0 && (i & 0x1) == 0)
	    CHECK(e1432_delete_channel_group(hw, group));
    }

    /* Create a group for subsequent tests */
    group = e1432_create_channel_group(hw, 0, NULL);
    if (group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, group);
	return -1;
    }

    if (verbose > 0)
	(void) printf("Testing e1432_channel_group_add\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_channel_group_add(hw, group, chan_list[1]));
    if (verbose > 0)
	(void) printf("Testing e1432_channel_group_remove\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_channel_group_remove(hw, group, chan_list[1]));

    CHECK(e1432_display_state(hw));

    if (verbose > 0)
	(void) printf("Testing e1432_delete_all_chan_groups\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_delete_all_chan_groups(hw));
    if (verbose > 0)
	(void) printf("Testing e1432_uninit_io_driver\n");
    for (i = 0; i < LOOPS; i++)
	CHECK(e1432_uninit_io_driver());

    if (verbose > 0)
	(void) printf("Testing multiple functions\n");
    for (i = 0; i < 128; i++)
    {
	CHECK(e1432_init_io_driver());
	CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));
	CHECK(e1432_assign_channel_numbers(nmod, laddr, &hw));
	group = e1432_create_channel_group(hw, 0, NULL);
	group = e1432_create_channel_group(hw, 4, chan_list);
	if (group >= 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_create_channel_group: returned %d\n",
			   progname, group);
	    return -1;
	}
	CHECK(e1432_channel_group_add(hw, group, chan_list[2]));
	CHECK(e1432_channel_group_remove(hw, group, chan_list[1]));

	if (i > 64)
	{
	    /* Empty the group */
	    CHECK(e1432_channel_group_remove(hw, group, chan_list[0]));
	    CHECK(e1432_channel_group_remove(hw, group, chan_list[1]));
	    CHECK(e1432_channel_group_remove(hw, group, chan_list[2]));
	    CHECK(e1432_channel_group_remove(hw, group, chan_list[3]));
	}

	/* We have a group.  Do something with it. */
	switch (i & 0xf)
	{
	case 0:
	    if (verbose > 1)
		(void) printf("Using e1432_delete_channel_group\n");
	    CHECK(e1432_delete_channel_group(hw, group));
	    break;
	case 1:
	    if (verbose > 1)
		(void) printf("Using e1432_delete_all_chan_groups\n");
	    CHECK(e1432_delete_all_chan_groups(hw));
	    /*FALLTHROUGH*/
	case 2:
	    if (verbose > 1)
		(void) printf("Using e1432_delete_all_chan_groups\n");
	    CHECK(e1432_delete_all_chan_groups(hw));
	    break;
	case 3:
	    if (verbose > 1)
		(void) printf("Using e1432_init_measure\n");
	    CHECK(e1432_init_measure(hw, group));
	    /*FALLTHROUGH*/
	case 4:
	    if (verbose > 1)
		(void) printf("Using e1432_init_measure\n");
	    CHECK(e1432_init_measure(hw, group));
	    break;
	case 5:
	    if (verbose > 1)
		(void) printf("Using e1432_init_measure_to_booted\n");
	    CHECK(e1432_init_measure_to_booted(hw, group, 0));
	    /*FALLTHROUGH*/
	case 6:
	    if (verbose > 1)
		(void) printf("Using e1432_init_measure_to_booted and finish\n");
	    CHECK(e1432_init_measure_to_booted(hw, group, 0));
	    CHECK(e1432_init_measure_finish(hw, group, 1));
	    break;
	case 7:
	    if (verbose > 1)
		(void) printf("Using e1432_init_measure of channel\n");
	    CHECK(e1432_init_measure(hw, chan_list[0]));
	    break;
	case 8:
	    if (verbose > 1)
		(void) printf("Using e1432_reset_measure\n");
	    CHECK(e1432_reset_measure(hw, group));
	    break;
	case 9:
	    if (verbose > 1)
		(void) printf("Using e1432_reset_measure of channel\n");
	    CHECK(e1432_reset_measure(hw, chan_list[0]));
	    break;
	case 10:
	    if (verbose > 1)
		(void) printf("Using e1432_finish_measure\n");
	    CHECK(e1432_finish_measure(hw, group));
	    break;
	case 11:
	    if (verbose > 1)
		(void) printf("Using e1432_finish_measure of channel\n");
	    CHECK(e1432_finish_measure(hw, chan_list[0]));
	    break;
	case 12:
	    if (verbose > 1)
		(void) printf("Using e1432_auto_zero\n");
	    CHECK(e1432_auto_zero(hw, group));
	    break;
	case 13:
	    if (verbose > 1)
		(void) printf("Using e1432_auto_zero of channel\n");
	    CHECK(e1432_auto_zero(hw, chan_list[0]));
	    break;
	case 14:
	    if (verbose > 1)
		(void) printf("Using e1432_auto_range\n");
	    CHECK(e1432_auto_range(hw, group, 0));
	    break;
	case 15:
	    if (verbose > 1)
		(void) printf("Using e1432_auto_range of channel\n");
	    CHECK(e1432_auto_range(hw, chan_list[0], 0));
	    break;
	}

	CHECK(e1432_uninit_io_driver());
    }

    CHECK(e1432_display_state(0));

    return 0;
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list)
{
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_init_measure(hw, group));

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, double clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 1)
	(void) printf("Block available found\n");

    return 0;
}

static int
get_scan(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 double clock_freq, int fft, int verbose, double span)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    int     type, chan;

    /* Wait for block available */
    CHECK(wait_block_avail(hw, group, verbose, BLOCKSIZE, span));

    /* Read the data */
    for (type = 0; type < (fft ? 2 : 1); type++)
	for (chan = 0; chan < nchan; chan++)
	{
	    if (verbose > 3)
		(void) printf("Reading chan %d type %d\n", chan, type);

	    CHECK(e1432_read_float64_data(hw, chan_list[chan],
					  type == 0 ? E1432_TIME_DATA :
					  E1432_FREQ_DATA, buffer,
					  BLOCKSIZE, &trailer,
					  &count));
	    if (count != BLOCKSIZE)
	    {
		(void) fprintf(stderr, "%s: e1432_read_float64_data: "
			       "actual count was %ld, expected %d\n",
			       progname, count, BLOCKSIZE);
		return -1;
	    }

	    CHECK(check_trailer(&trailer, clock_freq, span,
				chan_list[chan], type));
	}

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int nscan, int verbose)
{
    int     scan;

    for (scan = 0; scan < nscan; scan++)
	CHECK(get_scan(hw, group, nchan, chan_list,
		       51200, 0, verbose, 20000));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    int     c, i, nmod, group, nchan, verbose;
    char   *p;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:n:N:uvV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Stress the library a little */
    if (stress(nmod, laddr, verbose) < 0)
	return EXIT_FAILURE;

    /* Now make sure a normal measurement still runs */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, 3, verbose) < 0)
	return EXIT_FAILURE;

    /* A channel ID instead of a group ID should work */
    if (nchan > 0)
    {
	nchan = 1;
	if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	    return EXIT_FAILURE;
	if (setup(hw, chan_list[0], nchan, chan_list) < 0)
	    return EXIT_FAILURE;
	if (run(hw, chan_list[0], nchan, chan_list, 3, verbose) < 0)
	    return EXIT_FAILURE;
    }

    /* Should be able to start (but not run) a zero-channel measurement */
    nchan = 0;
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
